/* karmadb.c - Karma DB parser 
 * (c) 2006 Bob Copeland.  GPL v2.  
 *
 * Layout of smalldb:
 *
 * 0000	le32	version?
 * 0004	le32	creation time
 * 0008	le32	some number
 * 000c	le32	some number
 * 0010	le32	number of properties (36)
 * 0014 le32	first property, type
 * 0018	le32	first property, some flag (required?)
 * 001c	le32	2nd property type, and so on
 * [...]
 * 0134	le32	size of following string
 * ...	char	a string of prop names, delimited by spaces
 * ...	le32	null
 * ...	le32	# of songs, I think
 * ...	le32	# of items, incl taxi files and playlists
 *
 * Then each property for each fid follows.  The layout depends on the 
 * type.  There are five types: ints, strings, blobs, int8s and int16s.  
 * Strings have a string table and each fid has an offset into the table.  
 * Blobs have a raw binary table and each fid has an offset into it.  It 
 * appears the player has to have the knowledge of the length of individual 
 * blob items (playlists and profiles).  The properties are in order of their 
 * definition.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <time.h>
#include "karmadb.h"
#include "hexdump.h"

static 
int _get_max_attribute(karma_db_t *db, int attribute)
{
	int max = 0;
	GList *list = db->objects;

	while (list)
	{
		int *array = (int *) list->data;

		if (array[attribute] > max) 
			max = array[attribute];
		list = g_list_next(list);
	}
	return max;
}

static 
void _get_item_buf(unsigned char **ptr, k_atype_t type, int *result)
{
	int x;
	short s;
	unsigned char *buf = *ptr;

	switch (type)
	{
		case K_TYPE_LE32:
		case K_TYPE_STRING:
		case K_TYPE_BLOB:
			x =	(buf[3] << 24) | 
				(buf[2] << 16) |
				(buf[1] <<  8) |
				(buf[0]);
			*result = x;
			*ptr += 4;
			break;
		case K_TYPE_CHAR:
			*result = buf[0];
			*ptr += 1;
			break;
		case K_TYPE_LE16:
			s = (buf[1] <<  8) | buf[0];
			*result = s;
			*ptr += 2;
			break;
	}
}

static 
void _put_item(FILE *fp, k_atype_t type, int value)
{
	int x;
	short s;

	switch (type)
	{
		case K_TYPE_LE32:
		case K_TYPE_STRING:
		case K_TYPE_BLOB:
			fputc(value       & 0xff, fp);
			fputc(value >>  8 & 0xff, fp);
			fputc(value >> 16 & 0xff, fp);
			fputc(value >> 24 & 0xff, fp);
			break;
		case K_TYPE_CHAR:
			fputc(value, fp);
			break;
		case K_TYPE_LE16:
			fputc(value	  & 0xff, fp);
			fputc(value >>  8 & 0xff, fp);
	}
}

static 
void _get_item(FILE *fp, k_atype_t type, int *result)
{
	int x;
	short s;
	unsigned char buf[4];

	switch (type)
	{
		case K_TYPE_LE32:
		case K_TYPE_STRING:
		case K_TYPE_BLOB:
			fread(buf, 4, 1, fp);
			x =	(buf[3] << 24) | 
				(buf[2] << 16) |
				(buf[1] <<  8) |
				(buf[0]);
			*result = x;
			break;
		case K_TYPE_CHAR:
			*result = fgetc(fp);
			break;
		case K_TYPE_LE16:
			fread(buf, 2, 1, fp);
			s = (buf[1] <<  8) | buf[0];
			*result = s;
			break;
	}
}

static 
int is_deleted(karma_db_t *db, int index)
{
	if ((index >> 3) > db->dmap_size)
		return 0;
	return db->dmap[index >> 3] & (1 << (index % 8));
}


/*
 *  Get an entry given a fid
 */
karma_entry_t *kdb_find_entry(karma_db_t *db, int fid)
{
	GList *list = db->objects;
	char *key;
	int fidi;
	int entry_ind = 0;
	
	if (!g_hash_table_lookup_extended(db->name_hash, K_FID, 
				(gpointer) &key, 
				(gpointer) &fidi))
		return NULL;

	while (list)
	{
		int *array = (int *) list->data;
		if (array[fidi] == fid)
		{
			if (is_deleted(db, entry_ind))
				return NULL;

			return list->data;
		}

		list = g_list_next(list);
		entry_ind++;
	}
	return NULL;
}

static 
int _compare(const void *a, const void *b)
{
	struct k_str_info *ks1 = (struct k_str_info *) a;
	struct k_str_info *ks2 = (struct k_str_info *) b;
	return strcmp(ks1->str, ks2->str);
}

/*
 * Sorts the contents of a string table, and remaps all of the indices of
 * the attribute owners to the new indices.  Not optimal but it does the
 * job.  A better approach would be to store the strings as a tree or hash
 * in the first place.
 */
static 
void _sort_string_table(karma_db_t *db, int atbi)
{
	int i, j, *remap;
	int num_strings = 0;
	char *ptr, *p2;
	GList *list;
	k_attrib_t *attrib = &db->attribs[atbi];

	struct k_str_info *str_table;

	str_table = g_malloc(sizeof(struct k_str_info) * attrib->table_len);
	remap = g_malloc(sizeof(int *) * attrib->table_len);

	for (i=0, j=0; i<attrib->table_len; 
			i += strlen(&attrib->table[i]) + 1, j++)
	{
		str_table[j].str = &attrib->table[i];
		str_table[j].offset = i;
	}
	num_strings = j;

	qsort(str_table, num_strings, sizeof(struct k_str_info), _compare);

	/* build new compressed string */
	ptr = g_malloc(attrib->table_len);
	p2 = ptr;
	for (i=0; i < num_strings; i++)
	{
		strcpy(p2, str_table[i].str);
		remap[str_table[i].offset] = p2 - ptr;
		p2 += strlen(p2) + 1;
	}

	/* renumber the old attribs */
	for (list = db->objects; list; list = g_list_next(list))
	{
		int *iptr = (int *) list->data;
		iptr[atbi] = remap[iptr[atbi]];
	}

	g_free(attrib->table);
	attrib->table = ptr;
}

static
int _add_string(k_attrib_t *attrib, const char *str)
{
	int i;

	for (i=0; i<attrib->table_len; i += strlen (&attrib->table[i]) + 1)
	{
		if (strcmp(&attrib->table[i], str) == 0)
			return i;
	}

	attrib->table = g_realloc(attrib->table, 
		attrib->table_len + strlen(str) + 1);

	strcpy(attrib->table + attrib->table_len, str);
	attrib->table_len += strlen(str) + 1;

	return i;
}

int kdb_set_attribute(karma_db_t *db, karma_entry_t *entry, char *attrib,
		void *value, int len)
{
	int *record = (int *) entry;
	int i;
	char *key;

	if (!g_hash_table_lookup_extended(db->name_hash, attrib, 
				(gpointer) &key, 
				(gpointer) &i))
		return 0;
	
	if (i < 0 || i > db->num_attribs)
		return 0;

	switch (db->attribs[i].type)
	{
		case K_TYPE_LE32:
		case K_TYPE_LE16:
		case K_TYPE_CHAR:
			record[i] = (int) value;
			break;
		case K_TYPE_STRING:
			record[i] = _add_string(&db->attribs[i], value);
			break;
		case K_TYPE_BLOB:
			db->attribs[i].table = g_realloc(db->attribs[i].table, 
				db->attribs[i].table_len + len);
			memcpy(db->attribs[i].table + db->attribs[i].table_len,
				value, len);
			record[i] = db->attribs[i].table_len;
			db->attribs[i].table_len += len;
	}
	return 1;
}

/*
 *  Get an attribute value.  The returned pointer must not be altered.
 */
int kdb_get_attribute(karma_db_t *db, karma_entry_t *entry, char *attrib, 
			void* ret_ptr)
{
	int *record = (int *) entry;
	int i;
	char *key;

	if (!g_hash_table_lookup_extended(db->name_hash, attrib, 
				(gpointer) &key, 
				(gpointer) &i))
		return 0;
	
	if (i < 0 || i > db->num_attribs)
		return 0;

	switch (db->attribs[i].type)
	{
		case K_TYPE_LE32:
		case K_TYPE_LE16:
		case K_TYPE_CHAR:
			*(int *)ret_ptr = record[i];
			break;
		case K_TYPE_BLOB:
		case K_TYPE_STRING:
			*(char **)ret_ptr = &db->attribs[i].table[record[i]];
			break;
	}
	return 1;
}

int kdb_get_max_fid(karma_db_t *db)
{
	int fidi;
	char *key;

	if (!g_hash_table_lookup_extended(db->name_hash, K_FID, 
				(gpointer) &key, 
				(gpointer) &fidi))
		return -1;

	return _get_max_attribute(db, fidi);
}

void kdb_write(karma_db_t *db, FILE *fp)
{
	int i, len;

	db->ctime = (int) time(NULL);

	_put_item(fp, K_TYPE_LE32, db->version);
	_put_item(fp, K_TYPE_LE32, db->ctime);
	_put_item(fp, K_TYPE_LE32, db->eight);
	_put_item(fp, K_TYPE_LE32, db->three);
	_put_item(fp, K_TYPE_LE32, db->num_attribs);

	for (i=0, len=0; i < db->num_attribs; i++)
	{
		k_attrib_t *atb = &db->attribs[i];

		_put_item(fp, K_TYPE_LE32, atb->type);
		_put_item(fp, K_TYPE_LE32, atb->dunno);
		len += strlen(atb->name) + 1;
	}
	_put_item(fp, K_TYPE_LE32, len);
	for (i=0, len=0; i < db->num_attribs; i++)
	{
		char *name = db->attribs[i].name;
		fwrite(name, strlen(name), 1, fp);
		fputc(' ', fp);
	}
	_put_item(fp, K_TYPE_LE32, 0);
	_put_item(fp, K_TYPE_LE32, db->num_objects);  // # of.. tunes?
	_put_item(fp, K_TYPE_LE32, db->num_objects); 

	for (i=0; i < db->num_attribs && db->num_objects > 0; i++)
	{
		GList *list = db->objects;

		if (db->attribs[i].type == K_TYPE_STRING)
		{
			_put_item(fp, K_TYPE_LE32, db->attribs[i].table_len);
			_sort_string_table(db, i);
		}
		
		while (list)
		{
			int *iptr = (int *) list->data;

			_put_item(fp, db->attribs[i].type, iptr[i]);
			list = g_list_next(list);
		}

		if (db->attribs[i].type == K_TYPE_BLOB)
		{
			_put_item(fp, K_TYPE_LE32, db->attribs[i].table_len);
		}

		if ((db->attribs[i].type == K_TYPE_BLOB || 
		    db->attribs[i].type == K_TYPE_STRING) && 
		    db->attribs[i].table_len > 0)

		{
			fwrite(db->attribs[i].table, 1, 
			       db->attribs[i].table_len, fp);
		}
	}

	// read deleted map
	_put_item(fp, K_TYPE_LE32, db->dmap_size);

	if (db->dmap_size)
	{
		fwrite(db->dmap, 1, db->dmap_size, fp);
	}
	_put_item(fp, K_TYPE_LE32, db->ctime);

	fflush(fp);
}

karma_db_t *kdb_read(FILE *fp)
{
	karma_db_t *db;
	int i, dummy, len, check_date;
	char **vec, **ptr, *buf;

	db = g_malloc(sizeof(karma_db_t));

	db->objects = NULL;

	_get_item(fp, K_TYPE_LE32, &db->version);
	_get_item(fp, K_TYPE_LE32, &db->ctime);
	_get_item(fp, K_TYPE_LE32, &db->eight);
	_get_item(fp, K_TYPE_LE32, &db->three);
	_get_item(fp, K_TYPE_LE32, &db->num_attribs);

	db->attribs = g_malloc(db->num_attribs * sizeof(k_attrib_t));
	db->name_hash = g_hash_table_new(g_str_hash, g_int_equal);
	for (i=0; i < db->num_attribs; i++)
	{
		k_attrib_t *atb = &db->attribs[i];

		_get_item(fp, K_TYPE_LE32, (int *) &atb->type);
		_get_item(fp, K_TYPE_LE32, &atb->dunno);
	}

	_get_item(fp, K_TYPE_LE32, &len);
	buf = g_malloc(len);
	fread(buf, 1, len, fp);

	vec = g_strsplit(buf, " ", -1);

	for (i=0, ptr = vec; i < db->num_attribs && *ptr; i++)
	{
		db->attribs[i].name = *ptr++;
		db->attribs[i].table = NULL;
		g_hash_table_insert(db->name_hash, db->attribs[i].name, (gpointer)i);
	}
	g_free(vec);
	g_free(buf);

	_get_item(fp, K_TYPE_LE32, &dummy);  // null
	if (dummy != 0)
	{
		unsigned char buf[1024];
		unsigned char *x = buf;
		int to_read = 0;
		unsigned char *eptr;
		long start;

		printf("count?: %d\n", dummy);
		_get_item(fp, K_TYPE_LE32, &dummy);
		printf("object index: %d\n", dummy);
		_get_item(fp, K_TYPE_LE32, &to_read);
		printf("size data: %d\n", to_read);

		eptr = &buf[to_read];

		fread(buf, 1, to_read, fp);
#if 0
		// don't know how this works.
		for (i=0; i < db->num_attribs; i++)
		{
			unsigned char *ox = x;
			int foo;
			int plen;
			len = 0;

			if (db->attribs[i].type == K_TYPE_STRING)
			{
				printf("%s %s\n", db->attribs[i].name, x);
				x += strlen(x) + 1;
			}
			else 
			{
				_get_item_buf(&x, db->attribs[i].type, &foo);
				printf("%s %08x\n", db->attribs[i].name, foo);

				if (strcmp(db->attribs[i].name, "length") == 0)
					plen = foo;
			}

			if (db->attribs[i].type == K_TYPE_BLOB)
			{
				int count = plen;;
				if (db->attribs[i].name[1] == 'r')
					count = 40; 
				int j;
				for (j=0; j<count; j++) 
				{
				_get_item_buf(&x, K_TYPE_CHAR, &foo);
				printf("\tx: %02x\n", foo);
				}
			}
			/*
			if (len && db->attribs[i].type == K_TYPE_BLOB)
			{
				x += len;
			}
				*/
			hexdump(ox, eptr-ox, 1, 0);
		}
		printf("\n");
#endif
	}
	_get_item(fp, K_TYPE_LE32, &dummy);  // # of.. tunes?

	_get_item(fp, K_TYPE_LE32, &db->num_objects); 
	printf("%d %d\n", dummy, db->num_objects);

	for (i=0; i < db->num_objects; i++)
	{
		int *array = g_malloc(sizeof(int) * db->num_attribs);
		db->objects = g_list_prepend(db->objects, array);
	}

	for (i=0; i < db->num_attribs && db->num_objects > 0; i++)
	{
		GList *list = db->objects;

		if (db->attribs[i].type == K_TYPE_STRING)
		{
			_get_item(fp, K_TYPE_LE32, &len);
			db->attribs[i].table_len = len;
		}
		
		while (list)
		{
			int *iptr = (int *) list->data;

			_get_item(fp, db->attribs[i].type, &iptr[i]);
			list = g_list_next(list);
		}
		// No idea why they did it this way.
		if (db->attribs[i].type == K_TYPE_BLOB)
		{
			_get_item(fp, K_TYPE_LE32, &len);
			db->attribs[i].table_len = len;
		}

		if ((db->attribs[i].type == K_TYPE_BLOB || 
		    db->attribs[i].type == K_TYPE_STRING) && len)
		{
			db->attribs[i].table = g_malloc(len);
			fread(db->attribs[i].table, 1, len, fp);
		}
	}

	// read deleted map
	_get_item(fp, K_TYPE_LE32, &db->dmap_size);
	printf("size is %d\n", db->dmap_size);

	if (db->dmap_size)
	{
		db->dmap = g_malloc(db->dmap_size);
		fread(db->dmap, 1, db->dmap_size, fp);
	}

	_get_item(fp, K_TYPE_LE32, &check_date);
	if (check_date != db->ctime)
	{
		fprintf(stderr, "DB dates do not match (Corrupt DB?)\n");
		kdb_free(db);
		return NULL;
	}
	return db;
}

void kdb_free(karma_db_t *db)
{
	int i; GList *list;

	g_hash_table_destroy(db->name_hash);
	for (i=0; i < db->num_attribs; i++)
	{
		g_free(db->attribs[i].name);
		g_free(db->attribs[i].table);
	}
	for (list = db->objects; list; list=g_list_next(list))
		g_free(list->data);

	g_list_free(db->objects);
	g_free(db->attribs);
	g_free(db);
}

static 
void _print_utf8(char *s, int len)
{
	int i;
	// UTF-8 with certain characters escaped
	for (i=0; i<len; i++)
	{
		unsigned char c = s[i];

		if (c == '\\')
		        printf("%s", "\\\\"); 
		else if (c == '\n')
		        printf("%s", "\\n"); 
		else if (c & 0x80)
		{
			putchar(0xc0 | c >> 6);
			putchar(0x80 | c & 0x3f);
		}
		else
			if (c>=32)
				putchar(c);
			else
				printf("\\x%02hhX", c);
	}
}

static 
void _format_key_value(int i, k_attrib_t *attrib, int value, int len)
{
	printf("%-30s ", attrib->name);

	if (strcmp(attrib->name, K_FID) == 0)
	{
		printf("0x%.8x (%d)\n", value, value);
		return;
	}

	switch (attrib->type)
	{
		case K_TYPE_LE32:
		case K_TYPE_CHAR:
		case K_TYPE_LE16:
			printf("%d", value);
			break;

		case K_TYPE_STRING:
			len = strlen(&attrib->table[value]);
			_print_utf8(&attrib->table[value], len);
			// fall through
		case K_TYPE_BLOB:
			break;
	}
	putchar('\n');
}

void kdb_print(karma_db_t *db)
{
	int i;
	time_t t;

	printf("version: %d\n", db->version);
	t = db->ctime;
	printf("create time: %s", ctime(&t));
	printf("eight: %d\n", db->eight);
	printf("three: %d\n", db->three);
	printf("num attributes: %d\n", db->num_attribs);
	printf("max fid: %.8x (%d)\n", kdb_get_max_fid(db),kdb_get_max_fid(db));

	for (i=0; i < db->num_attribs; i++)
	{
		printf("  %s %d %d\n", db->attribs[i].name, 
			db->attribs[i].type, db->attribs[i].dunno);
	}
	printf("num objects: %d\n", db->num_objects);
	printf("\n");

	GList *list = db->objects;
	int count = 0;
	while (list)
	{
		if (is_deleted(db, count++))
			printf("=== DELETED ===\n");

	        int len = 0;
		int *iptr = (int *) list->data;

		for (i=0; i < db->num_attribs; i++)
		{
		        k_attrib_t *attrib = &db->attribs[i];
			if (strcmp(attrib->name, "length") == 0)
				len = iptr[i];
		}
		for (i=0; i < db->num_attribs; i++)
		{
			int alen = 0;
		        k_attrib_t *attrib = &db->attribs[i];
		        if (strcmp(attrib->name, "profile") == 0)
				alen = 40;
			else if (strcmp(attrib->name, "playlist") == 0)
				alen = len;

  		        _format_key_value(i, &db->attribs[i], iptr[i], alen);
		}
		printf("\n");
		list = g_list_next(list);
	}
}

